from functools import partial

import torch
import torch.nn as nn

import timm
import timm.models.vision_transformer

from scheduler.time_embedder import TimestepEmbedder
new_timm = '0.9' in timm.__version__ 

class VisionTransformer(timm.models.vision_transformer.VisionTransformer):
    """ Vision Transformer with support for global average pooling
    """
    def __init__(self, global_pool=False, is_hybrid=False, **kwargs): 
        global_pool = "avg" if global_pool else "token"
        super(VisionTransformer, self).__init__(
            global_pool=global_pool,
            **kwargs
        )
        self.is_hybrid = is_hybrid
        if global_pool == "avg":
            norm_layer = kwargs['norm_layer']
            embed_dim = kwargs['embed_dim']
            self.fc_norm = norm_layer(embed_dim)

            self.norm = nn.Identity()  
        if self.is_hybrid:
            self.t_emb = TimestepEmbedder(embed_dim)

    def forward_features(self, x): 
        if new_timm:
            x = super().forward_features(x)
            return x
            
        B = x.shape[0]
        x = self.patch_embed(x)
        
        if self.is_hybrid:
            timesteps = torch.full((B,), 250, device=x.device).long()
            t = self.t_embed(timesteps, token_size=x.shape[1])
            
        cls_tokens = self.cls_token.expand(B, -1, -1)  
        x = torch.cat((cls_tokens, x), dim=1)
        
        x = x + self.pos_embed
        x = self.pos_drop(x)

        for idx, blk in enumerate(self.blocks):
            if self.is_hybrid and idx == 2:
                x = x + t
            x = blk(x)

        if self.global_pool == "avg":
            x = x[:, 1:, :].mean(dim=1)  
            outcome = self.fc_norm(x)
        else:
            x = self.norm(x)
            outcome = x[:, 0]

        return outcome

def vit_small_patch16(**kwargs):
    model = VisionTransformer(
        patch_size=16, embed_dim=384, depth=12, num_heads=6, mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), **kwargs)
    return model

def vit_base_patch16(**kwargs):
    model = VisionTransformer(
        patch_size=16, embed_dim=768, depth=12, num_heads=12, mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), **kwargs)
    return model

def vit_large_patch16(**kwargs):
    model = VisionTransformer(
        patch_size=16, embed_dim=1024, depth=24, num_heads=16, mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), **kwargs)
    return model

def vit_huge_patch14(**kwargs):
    model = VisionTransformer(
        patch_size=14, embed_dim=1280, depth=32, num_heads=16, mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), **kwargs)
    return model